import { useState } from 'react'
import { useNavigate } from 'react-router-dom'
import { Calendar, CheckCircle, Scale, Users, Home, Briefcase, FileText, Sparkles, Award, Shield, Clock, TrendingUp, User } from 'lucide-react'
import { motion } from 'framer-motion'
import './NewConsultation.css'

const NewConsultation = () => {
  const navigate = useNavigate()
  const [selectedArea, setSelectedArea] = useState('')
  const [selectedDate, setSelectedDate] = useState('')
  const [isSubmitted, setIsSubmitted] = useState(false)

  const areas = [
    { 
      id: 'family', 
      name: 'Семейное право', 
      icon: Users,
      color: '#1E40AF',
      description: 'Разводы, алименты, опека',
      gradient: 'linear-gradient(135deg, #1E40AF 0%, #3B82F6 100%)',
      duration: '1-1.5 часа',
      price: 'от 3000₽'
    },
    { 
      id: 'labor', 
      name: 'Трудовое право', 
      icon: Briefcase,
      color: '#3B82F6',
      description: 'Трудовые споры, увольнения',
      gradient: 'linear-gradient(135deg, #3B82F6 0%, #60A5FA 100%)',
      duration: '1-2 часа',
      price: 'от 3500₽'
    },
    { 
      id: 'housing', 
      name: 'Жилищное право', 
      icon: Home,
      color: '#60A5FA',
      description: 'Сделки с недвижимостью',
      gradient: 'linear-gradient(135deg, #60A5FA 0%, #3B82F6 100%)',
      duration: '1.5-2 часа',
      price: 'от 4000₽'
    },
    { 
      id: 'criminal', 
      name: 'Уголовное право', 
      icon: FileText,
      color: '#2563EB',
      description: 'Защита по уголовным делам',
      gradient: 'linear-gradient(135deg, #2563EB 0%, #1E40AF 100%)',
      duration: '2-3 часа',
      price: 'от 5000₽'
    }
  ]

  const lawyers = [
    { name: 'Анна Соколова', experience: '15 лет', rating: 5, specialty: 'Семейное, Жилищное право', cases: '500+' },
    { name: 'Михаил Петров', experience: '12 лет', rating: 5, specialty: 'Трудовое, Корпоративное право', cases: '400+' },
    { name: 'Елена Волкова', experience: '18 лет', rating: 5, specialty: 'Уголовное, Административное право', cases: '600+' },
    { name: 'Дмитрий Иванов', experience: '10 лет', rating: 5, specialty: 'Жилищное, Семейное право', cases: '350+' }
  ]

  const guarantees = [
    { icon: Shield, title: 'Конфиденциальность', text: 'Полная защита персональных данных' },
    { icon: Award, title: 'Профессионализм', text: 'Только опытные специалисты' },
    { icon: Clock, title: 'Оперативность', text: 'Быстрые ответы на ваши вопросы' },
    { icon: TrendingUp, title: 'Результативность', text: '95% успешных решений дел' }
  ]

  const handleSubmit = (e) => {
    e.preventDefault()
    if (selectedArea && selectedDate) {
      const savedConsultations = localStorage.getItem('lawConsultations')
      const consultations = savedConsultations ? JSON.parse(savedConsultations) : []
      
      const areaData = areas.find(a => a.id === selectedArea)
      const randomLawyer = lawyers[Math.floor(Math.random() * lawyers.length)]
      
      const newConsultation = {
        id: Date.now(),
        topic: areaData.name,
        lawyer: randomLawyer.name,
        status: 'В процессе',
        review: null
      }

      consultations.push(newConsultation)
      localStorage.setItem('lawConsultations', JSON.stringify(consultations))
      
      setIsSubmitted(true)
      setTimeout(() => {
        setIsSubmitted(false)
        setSelectedArea('')
        setSelectedDate('')
        navigate('/')
      }, 2000)
    }
  }

  const minDate = new Date().toISOString().split('T')[0]
  const maxDate = new Date(Date.now() + 90 * 24 * 60 * 60 * 1000).toISOString().split('T')[0]

  return (
    <div className="new-consultation-page">
      <div className="container">
        <motion.section 
          className="consultation-hero"
          initial={{ opacity: 0, y: 30 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6 }}
        >
          <motion.div 
            className="consultation-icon-wrapper"
            initial={{ scale: 0, rotate: 180 }}
            animate={{ scale: 1, rotate: 0 }}
            transition={{ type: "spring", stiffness: 200, damping: 15, delay: 0.3 }}
          >
            <Sparkles size={80} strokeWidth={2.5} className="consultation-hero-icon" />
          </motion.div>
          <h1 className="consultation-title">Новая консультация</h1>
          <p className="consultation-subtitle">
            Выберите область права и удобную дату для получения профессиональной юридической помощи
          </p>
        </motion.section>

        <motion.section 
          className="consultation-form-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.2 }}
        >
          <div className="consultation-form-wrapper">
            <form className="consultation-form" onSubmit={handleSubmit}>
              <div className="form-group">
                <label className="form-label">
                  <Scale size={26} strokeWidth={2.5} className="label-icon" />
                  Выберите область права
                </label>
                <div className="areas-grid">
                  {areas.map((area) => {
                    const IconComponent = area.icon
                    return (
                      <motion.button
                        key={area.id}
                        type="button"
                        className={`area-card ${selectedArea === area.id ? 'selected' : ''}`}
                        onClick={() => setSelectedArea(area.id)}
                        whileHover={{ scale: 1.08, rotate: 3 }}
                        whileTap={{ scale: 0.98 }}
                        style={{ 
                          borderColor: selectedArea === area.id ? area.color : 'rgba(30, 64, 175, 0.25)',
                          background: selectedArea === area.id ? (area.color + '18') : 'rgba(255, 255, 255, 0.98)'
                        }}
                      >
                        <div className="area-icon-wrapper" style={{ background: area.color + '25' }}>
                          <IconComponent size={44} strokeWidth={2.5} color={area.color} />
                        </div>
                        <div className="area-content">
                          <h3 className="area-name">{area.name}</h3>
                          <p className="area-description">{area.description}</p>
                          <div className="area-meta">
                            <span className="area-duration">{area.duration}</span>
                            <span className="area-price">{area.price}</span>
                          </div>
                        </div>
                        {selectedArea === area.id && (
                          <motion.div 
                            className="area-check"
                            initial={{ scale: 0 }}
                            animate={{ scale: 1 }}
                            style={{ background: area.gradient }}
                          >
                            <CheckCircle size={28} strokeWidth={2.5} color="white" />
                          </motion.div>
                        )}
                      </motion.button>
                    )
                  })}
                </div>
              </div>

              <div className="form-group">
                <label htmlFor="date" className="form-label">
                  <Calendar size={26} strokeWidth={2.5} className="label-icon" />
                  Дата консультации
                </label>
                <input
                  type="date"
                  id="date"
                  className="form-input"
                  value={selectedDate}
                  onChange={(e) => setSelectedDate(e.target.value)}
                  min={minDate}
                  max={maxDate}
                  required
                />
              </div>

              <motion.button 
                type="submit" 
                className="submit-button" 
                disabled={!selectedArea || !selectedDate}
                whileHover={{ scale: 1.03, y: -4 }}
                whileTap={{ scale: 0.98 }}
              >
                <span>Записаться на консультацию</span>
                <CheckCircle size={26} strokeWidth={2.5} className="btn-icon" />
              </motion.button>

              {isSubmitted && (
                <motion.div 
                  className="success-message"
                  initial={{ opacity: 0, scale: 0.9 }}
                  animate={{ opacity: 1, scale: 1 }}
                  transition={{ duration: 0.3 }}
                >
                  <CheckCircle size={32} strokeWidth={2.5} className="success-icon" />
                  <span>Вы успешно записались на консультацию!</span>
                </motion.div>
              )}
            </form>
          </div>
        </motion.section>

        <motion.section 
          className="lawyers-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.4 }}
        >
          <h2 className="section-title">Наши юристы</h2>
          <div className="lawyers-grid">
            {lawyers.map((lawyer, index) => (
              <motion.div
                key={lawyer.name}
                className="lawyer-card"
                initial={{ opacity: 0, scale: 0.9 }}
                animate={{ opacity: 1, scale: 1 }}
                transition={{ delay: index * 0.1, duration: 0.5 }}
                whileHover={{ scale: 1.08, rotate: 3 }}
              >
                <div className="lawyer-avatar">
                  <User size={44} strokeWidth={2.5} />
                </div>
                <h3 className="lawyer-name">{lawyer.name}</h3>
                <p className="lawyer-experience">{lawyer.experience} опыта</p>
                <div className="lawyer-rating">
                  {Array.from({ length: 5 }, (_, i) => (
                    <Award key={i} size={18} fill="#F59E0B" stroke="#F59E0B" strokeWidth={2} />
                  ))}
                </div>
                <p className="lawyer-specialty">{lawyer.specialty}</p>
                <div className="lawyer-cases">
                  <span className="cases-count">{lawyer.cases}</span>
                  <span className="cases-label">выигранных дел</span>
                </div>
              </motion.div>
            ))}
          </div>
        </motion.section>

        <motion.section 
          className="guarantees-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.5 }}
        >
          <h2 className="section-title">Наши гарантии</h2>
          <div className="guarantees-grid">
            {guarantees.map((guarantee, index) => {
              const IconComponent = guarantee.icon
              return (
                <motion.div
                  key={guarantee.title}
                  className="guarantee-card"
                  initial={{ opacity: 0, y: 20 }}
                  animate={{ opacity: 1, y: 0 }}
                  transition={{ delay: index * 0.1, duration: 0.5 }}
                  whileHover={{ translateY: -12, boxShadow: '0 16px 40px rgba(30, 64, 175, 0.25)' }}
                >
                  <div className="guarantee-icon">
                    <IconComponent size={36} strokeWidth={2.5} color="#1E40AF" />
                  </div>
                  <h3 className="guarantee-title">{guarantee.title}</h3>
                  <p className="guarantee-text">{guarantee.text}</p>
                </motion.div>
              )
            })}
          </div>
        </motion.section>
      </div>
    </div>
  )
}

export default NewConsultation

